<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Prison;
use Illuminate\Support\Facades\DB;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $users = User::with(['prison.quartier.arrondissement.zone.ville'])->get();

        $title="Liste de tout les utilisateurs";
        return view('users.index', compact('users',"title"));
    }
    public function cc(Request $request)
    {
        $users =User::with('prison.quartier.arrondissement.zone.ville')
                ->where('role', 'cc')
                ->get();

        $title="Commisaires centraux";
        
        return view('users.index', compact('users',"title"));
    }
    public function cpa(Request $request)
    {
        $users =User::with('prison.quartier.arrondissement.zone.ville')
                ->where('role', 'cpa')
                ->get();

        $title="Commisaires de police d'arrondissement";
        
        return view('users.index', compact('users',"title"));
    }
    public function cpq(Request $request)
    {
        $users =User::with('prison.quartier.arrondissement.zone.ville')
                ->where('role', 'cpq')
                ->get();

        $title="Commisaires de police de quartier";
        
        return view('users.index', compact('users',"title"));
    }
    public function cp(Request $request)
    {
        $users =User::with('prison.quartier.arrondissement.zone.ville')
                ->where('role', 'cp')
                ->get();

        $title="Chef de poste";
        
        return view('users.index', compact('users',"title"));
    }

    public function geolier(Request $request)
    {
        $users =User::with('prison.quartier.arrondissement.zone.ville')
                ->where('role', 'geolier')
                ->get();

        $title="Les géôliers";
        
        return view('users.index', compact('users',"title"));
    }
    public function interne(Request $request)
    {
        $search = $request->input('search');

        $users = User::where('type', 0) // <- filtrage par type
        ->when($search, function ($query, $search) {
            return $query->where(function ($q) use ($search) {
                $q->where('nom', 'like', '%' . $search . '%')
                ->orWhere('email', 'like', '%' . $search . '%')
                ->orWhere('telephone', 'like', '%' . $search . '%');
            });
        })
        ->paginate(10);

        $title="Liste des admins";

        return view('users.index', compact('users',"title"));
    }
   

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $prisons = Prison::all(); // On récupère toutes les prisons

        return view('users.create', compact('prisons'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {   
        $validated = $request->validate([
            'name' => 'required|string',
            'email' => 'nullable|email|unique:users',
            'phone_number' => 'required|string',
            'role' => 'required|string',
            'assigned_prison_id' => 'nullable|exists:prisons,id',
            'password' => 'required|string|min:6',
            'photo' => 'nullable|image|max:2048',
        ]);

        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('photos', 'public');
            $validated['photo'] = $path;

            //dd($validated['photo']);
        }

        $validated['password'] = Hash::make($validated['password']);

        User::create($validated);

        return redirect()->route('users.index')->with('success', 'Utilisateur ajouté avec succès.');
    }


    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = User::findOrFail($id);
        $prisons = Prison::all();

        return view('users.edit', compact('user', 'prisons'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:users,email,' . $user->id,
            'phone_number' => 'required|string|max:20',
            'role' => 'required|string',
            'assigned_prison_id' => 'nullable|exists:prisons,id',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone_number = $request->phone_number;
        $user->role = $request->role;
        $user->assigned_prison_id = $request->assigned_prison_id;

        // Gestion de la photo
        if ($request->hasFile('photo')) {
            if ($user->photo && \Storage::exists($user->photo)) {
                \Storage::delete($user->photo);
            }

            $user->photo = $request->file('photo')->store('photos', 'public');
        }

        $user->save();

        return redirect()->route('users.index')->with('success', 'Utilisateur mis à jour avec succès.');
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();
        return redirect()->route('users.index')->with('success', 'Utilisateur supprimé avec succès.');

    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'integer|exists:users,id',
        ]);

        User::whereIn('id', $request->ids)->delete();

        return redirect()->route('users.index')->with('success', 'Utilisateurs supprimés avec succès.');
    }

    public function stat()
    {
        $usersCount = User::count();

        $usersByRole = User::select('role', DB::raw('count(*) as total'))
            ->groupBy('role')
            ->pluck('total', 'role');

        $usersByQuartier = User::join('prisons', 'users.assigned_prison_id', '=', 'prisons.id')
            ->join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->select('quartiers.name as quartier', DB::raw('count(*) as total'))
            ->groupBy('quartier')
            ->pluck('total', 'quartier');

        $usersByZone = User::join('prisons', 'users.assigned_prison_id', '=', 'prisons.id')
            ->join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->join('arrondissements', 'quartiers.arrondissement_id', '=', 'arrondissements.id')
            ->join('zones', 'arrondissements.zone_id', '=', 'zones.id')
            ->select('zones.name as zone', DB::raw('count(*) as total'))
            ->groupBy('zone')
            ->pluck('total', 'zone');

        $usersByArrondissement = User::join('prisons', 'users.assigned_prison_id', '=', 'prisons.id')
            ->join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->join('arrondissements', 'quartiers.arrondissement_id', '=', 'arrondissements.id')
            ->select('arrondissements.name as arrondissement', DB::raw('count(*) as total'))
            ->groupBy('arrondissement')
            ->pluck('total', 'arrondissement');

        $usersByVille = User::join('prisons', 'users.assigned_prison_id', '=', 'prisons.id')
            ->join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->join('arrondissements', 'quartiers.arrondissement_id', '=', 'arrondissements.id')
            ->join('zones', 'arrondissements.zone_id', '=', 'zones.id')
            ->join('villes', 'zones.ville_id', '=', 'villes.id')
            ->select('villes.name as ville', DB::raw('count(*) as total'))
            ->groupBy('ville')
            ->pluck('total', 'ville');

        $usersByMonth = User::select(DB::raw("DATE_FORMAT(created_at, '%Y-%m') as month"), DB::raw('count(*) as total'))
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month');
        $title="Statistiques";
        return view('users.stat', compact(
            'usersCount', 'usersByRole', 'usersByQuartier', 'usersByZone', 'usersByArrondissement', 'usersByVille', 'usersByMonth','title'
        ));
    }

    

}
