<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Reservation;
use App\Models\Voiture;
use App\Models\User;
use App\Models\GammesVehicules;
use App\Models\Finance;

use Illuminate\Support\Facades\DB;


class ReservationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');

        $reservations = Reservation::where('statut', 0)
        ->when($search, function ($query, $search) {
            return $query->where(function ($q) use ($search) {
                $q->where('code', 'like', '%' . $search . '%')
                  ->orWhere('depart', 'like', '%' . $search . '%')
                  ->orWhere('destination', 'like', '%' . $search . '%')
                  ->orWhere('created_at', 'like', '%' . $search . '%')
                  ->orWhere('telephone', 'like', '%' . $search . '%');
            });
        })
        ->orderByDesc('id')
        ->paginate(50);

        $chauffeurs = User::where('type', 2)->get();

        $title = "Liste des réservations en attente";

        $etat=0;

        return view('reservations.index', compact('reservations', 'title','chauffeurs','etat'));
    }
    public function confirm(Request $request)
    {
        $search = $request->input('search');

        $reservations = Reservation::where('statut', 1)
        ->where('desservi', 0)
        ->when($search, function ($query, $search) {
            return $query->where(function ($q) use ($search) {
                $q->where('code', 'like', '%' . $search . '%')
                  ->orWhere('depart', 'like', '%' . $search . '%')
                  ->orWhere('destination', 'like', '%' . $search . '%')
                  ->orWhere('created_at', 'like', '%' . $search . '%')
                  ->orWhere('telephone', 'like', '%' . $search . '%');
            });
        })
        ->orderByDesc('id')
        ->paginate(50);

        $chauffeurs = User::where('type', 2)->get();

        $title = "Liste des réservations confirmées";

        $etat=1;

        return view('reservations.index', compact('reservations', 'title','chauffeurs','etat'));
    }
    public function desservi(Request $request)
    {
        $search = $request->input('search');

        $reservations = Reservation::where('desservi', 1)
        ->when($search, function ($query, $search) {
            return $query->where(function ($q) use ($search) {
                $q->where('code', 'like', '%' . $search . '%')
                  ->orWhere('depart', 'like', '%' . $search . '%')
                  ->orWhere('destination', 'like', '%' . $search . '%')
                  ->orWhere('created_at', 'like', '%' . $search . '%')
                  ->orWhere('telephone', 'like', '%' . $search . '%');
            });
        })
        ->orderByDesc('id')
        ->paginate(50);

        $chauffeurs = User::where('type', 2)->get();

        $title = "Liste des réservations confirmées et desservies";
        $etat=2;

        return view('reservations.index', compact('reservations', 'title','chauffeurs','etat'));
    }
    

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $gammes = GammesVehicules::all(); // Récupère toutes les gammes
        $clients = User::where('type', 1)->get();  // type 1 = client
        $chauffeurs = User::where('type', 2)->get(); // type 2 = chauffeur
        $cars = Voiture::all();

        return view('reservations.create', compact('chauffeurs', 'cars','clients','gammes'));
    }
    
    /**
     * Store a newly created resource in storage.
     */

     public static function generateUniqueCode()
    {
        do {
            $prefix = 'RES-' . now()->format('Ymd');  // Par exemple "RES-20230506"
            $randomString = strtoupper(bin2hex(random_bytes(2))); // 4 caractères aléatoires
            $code = $prefix . '-' . $randomString;
        } while (Reservation::where('code', $code)->exists());  // Vérifier si le code existe déjà

        return $code;
    }

     public function store(Request $request)
     {
        
         // Validation des données
         $request->validate([
             'depart' => 'required|string|max:255',
             'destination' => 'required|string|max:255',
             'date_depart' => 'required|date',
             'heure_depart' => 'required|string|max:10',
             'place' => 'required|integer',
             'user_id' => 'required|exists:users,id',
             'telephone' => 'required|string|max:50',
             'solde' => 'required|numeric',
             'id_chauffeur' => 'nullable|exists:users,id',
             'latitude' => 'nullable|numeric',
             'longitude' => 'nullable|numeric',
             'latitude1' => 'nullable|numeric',
             'longitude1' => 'nullable|numeric',
             'gamme_id' => 'nullable|exists:gammes_vehicules,id',
             'detail' => 'nullable|string|max:255',
             'distance' => 'nullable|string',
             'duration' => 'nullable|string',
         ]);
 
         // Création de la nouvelle réservation
         $reservation = new Reservation();
         $reservation->code = $this->generateUniqueCode();
         $reservation->depart = $request->input('depart');
         $reservation->destination = $request->input('destination');
         $reservation->date_depart = $request->input('date_depart');
         $reservation->heure_depart = $request->input('heure_depart');
         $reservation->place = $request->input('place');
         $reservation->id_user = $request->input('user_id');
         $reservation->telephone = $request->input('telephone');
         $reservation->solde = $request->input('solde');
         $reservation->statut = 0;
         $reservation->id_chauffeur = $request->input('id_chauffeur');
         $reservation->latitude1 = $request->input('latitude');
         $reservation->longitude1 = $request->input('longitude');
         $reservation->latitude2 = $request->input('latitude1');
         $reservation->longitude2 = $request->input('longitude1');
         $reservation->detail = $request->input('detail');
         $reservation->distance = $request->input('distance');
         $reservation->duration = $request->input('duration');
         $reservation->gamme_id = $request->input('gamme_id');
         $reservation->created_at = now();
 
         // Récupération de la gamme et mise à jour du tarif si disponible
         if ($request->has('gamme_id')) {
             $gamme = GammesVehicules::find($request->input('gamme_id'));
             if ($gamme) {
                 $reservation->gamme_libelle = $gamme->libelle;
                 $reservation->gamme_tarif = $gamme->tarif;
             }
         }
 
         // Sauvegarder la réservation
         $reservation->save();
 
         // Retourner à la liste des réservations avec un message de succès
         return redirect()->route('reservations.index')->with('success', 'Réservation créée avec succès.');
     }
    

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $reservation = Reservation::with(['user', 'chauffeur', 'car', 'gamme'])->findOrFail($id);

        $search = "";

        $finances = Finance::with('reservation')->where('reservation_id', $id)
            ->when($search, function ($query, $search) {
                $query->where('type', 'like', "%$search%")
                    ->orWhere('source', 'like', "%$search%")
                    ->orWhere('description', 'like', "%$search%");
            })
            ->orderByDesc('date_operation')
            ->paginate(15);


        return view('reservations.show', [
            'reservation'=>$reservation,
            'finances'=>$finances
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $reservation = Reservation::findOrFail($id);
        
        // Récupérer les clients (type 1) et les chauffeurs (type 2)
        $clients = User::where('type', 1)->get();
        $chauffeurs = User::where('type', 2)->get();

        // Récupérer toutes les gammes disponibles
        $gammes = GammesVehicules::all();

        return view('reservations.edit', compact('reservation', 'clients', 'chauffeurs', 'gammes'));
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $reservation = Reservation::findOrFail($id);

        $request->validate([
            'depart' => 'required|string',
            'destination' => 'required|string',
            'date_depart' => 'required|date',
            'heure_depart' => 'required',
            'user_id' => 'required|exists:users,id',
            'gamme_id' => 'required|exists:gammes_vehicules,id',
            'place' => 'required|integer|min:1',
            'telephone' => 'required|string',
            'solde' => 'required|numeric',
            'distance' => 'nullable|string',
            'duration' => 'nullable|string',
            'detail' => 'nullable|string',
            'latitude' => 'nullable|string',
            'longitude' => 'nullable|string',
            'latitude1' => 'nullable|string',
            'longitude1' => 'nullable|string',
        ]);

        $reservation->update([
            'depart' => $request->depart,
            'destination' => $request->destination,
            'date_depart' => $request->date_depart,
            'heure_depart' => $request->heure_depart,
            'id_user' => $request->user_id,
            'gamme_id' => $request->gamme_id,
            'place' => $request->place,
            'telephone' => $request->telephone,
            'solde' => (float) $request->solde,
            'distance' => $request->distance,
            'duration' => $request->duration,
            'detail' => $request->detail,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'latitude1' => $request->latitude1,
            'longitude1' => $request->longitude1,
        ]);

        return redirect()->route('reservations.index')->with('success', 'Réservation mise à jour avec succès.');
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $reservation = Reservation::findOrFail($id);
        $reservation->delete();

        return redirect()->route('reservations.index')->with('success', 'Réservation supprimée avec succès.');
    }

    public function bulkDelete(Request $request)
    {
        if($request->action_group==1){

           

            $request->validate([
                'ids' => 'required|array',
                'ids.*' => 'integer|exists:reservation,id',
                'id_chauffeur' => 'required|integer|exists:users,id',
            ]);

            //dd($request->id_chauffeur);

            Reservation::whereIn('id', $request->ids)->update([
                'statut' => 1,
                'id_chauffeur' => $request->id_chauffeur,
            ]);

            return redirect()->route('reservations.confirm')->with('success', 'Reservations affectées au chauffeur avec succès.');

        }
        elseif($request->action_group==2){
            $request->validate([
                'ids' => 'required|array',
                'ids.*' => 'integer|exists:reservation,id',
            ]);

            //dd($request->id_chauffeur);

            Reservation::whereIn('id', $request->ids)->update([
                'desservi' => 1
            ]);

            return redirect()->route('reservations.desservi')->with('success', 'Reservations desservies avec succès.');

        }
        else{
            $request->validate([
                'ids' => 'required|array',
                'ids.*' => 'integer|exists:reservation,id',
            ]);
    
            Reservation::whereIn('id', $request->ids)->delete();

            return redirect()->route('reservations.index')->with('success', 'Reservations supprimées avec succès.');
        }
    }

    public function statistique(){
        // 1. Réservations par mois
        $reservationsParMois = Reservation::select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as mois'),
                DB::raw('COUNT(*) as total')
            )
            ->groupBy('mois')
            ->orderBy('mois')
            ->get();

        // 2. Statut des réservations
        $reservationsParStatut = Reservation::select(
                'statut',
                DB::raw('COUNT(*) as total')
            )
            ->groupBy('statut')
            ->get();

        // 3. Revenus par mois
        $revenusParMois = Reservation::select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as mois'),
                DB::raw('SUM(solde) as revenus')
            )
            ->groupBy('mois')
            ->orderBy('mois')
            ->get();

        // 4. Répartition des gammes
        $gammes = Reservation::select(
                'gamme_libelle',
                DB::raw('COUNT(*) as total')
            )
            ->groupBy('gamme_libelle')
            ->get();

        // 5. Réservations desservies vs non desservies
        $desservis = Reservation::select(
            DB::raw('CASE WHEN desservi = 1 THEN "Desservi" ELSE "Non desservi" END as statut_desservi'),
            DB::raw('COUNT(*) as total')
        )
        ->groupBy('statut_desservi')
        ->get();
        
        $totalReservations = Reservation::count();
        $totalDesservies = Reservation::where('desservi', 1)->count();
        $totalNonDesservies = Reservation::where('desservi', 0)->count();
        $revenuTotal = Reservation::sum('solde');

        return view('reservations.stat', compact(
            'reservationsParMois',
            'reservationsParStatut',
            'revenusParMois',
            'gammes',
            'desservis',
            'totalReservations',
            'totalDesservies',
            'totalNonDesservies',
            'revenuTotal'
        ));
    }
}
