<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Quartier;
use App\Models\Prison;
use App\Models\Prisonier;
use App\Models\CrimePrisonier;
use App\Models\Arrestation;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;


class PrisonierController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $prisonniers = Prisonier::with('prison')->get();
        $title = "Liste des prisonniers";

        return view('prisoniers.index', compact('prisonniers', 'title'));
    }

    public function prisonier_arrestation()
    {
       $arrestations = Arrestation::with('prison', 'user', 'crimes')->get();

        $title = "Liste des arrestations";

        return view('prisoniers.arrestations.index', compact('title', 'arrestations'));
    }

    function generateUniqueMatricule($length = 6)
    {
        do {
            $letters = strtoupper(Str::random(2)); // lettres A-Z
            $numbers = str_pad(random_int(0, 9999), 4, '0', STR_PAD_LEFT); // nombres sur 4 chiffres
            $matricule = $letters . $numbers;
        } while (Prisonier::where('matricule', $matricule)->exists());
        return $matricule;
    }

    // fonction create
    public function create()
    {
        $title="Nouveau prisonier";
        $matricule=$this->generateUniqueMatricule();
        return view('prisoniers.create', compact('title','matricule'));
    }
    public function prisonier_arrestation_create(Request $request)
    {
        $id=0;
        if($request->has('prisonier_id')){
            $id=$request->prisonier_id;
        }
        $title="Nouvelle arrestation";
        $prisoniers=Prisonier::all();
        $prisons=Prison::all();
        $crime_prisoniers=CrimePrisonier::where('prisonnier_id','=',$id)->get();
        $users=User::all();

        return view('prisoniers.arrestations.create', compact('title','prisoniers','prisons','id','crime_prisoniers','users'));
    }
    public function prisonier_arrestation_create_prisonier($id)
    {
        $title="Nouvelle arrestation";
        $prisoniers=Prisonier::all();

        return view('prisoniers.arrestations.create', compact('title','prisoniers','id'));
    }

    public function prisonier_arrestation_store(Request $request)
    {
        // Validation des données
        $request->validate([
            'date_arrestation' => 'required|date',
            'prison_id' => 'required|exists:prisons,id',
            'prisonier_id' => 'required|exists:prisoniers,id',
            'responsable_id' => 'required|exists:users,id',
            'statut' => 'required|string',
            'date_relachement' => 'nullable|date',
            'crimes' => 'required|array',
            'crimes.*' => 'exists:crimes,id',
        ]);

        // Création de l'arrestation
        $arrestation = Arrestation::create([
            'responsable_id' => $request->responsable_id,
            'prisonier_id' => $request->prisonier_id,
            'prison_id' => $request->prison_id,
            'date_arrestation' => $request->date_arrestation,
            'statut' => $request->statut,
            'date_relachement' => $request->date_relachement,
        ]);

        // Attacher les crimes à l'arrestation
        $arrestation->crimes()->attach($request->crimes);

        return redirect()->route('prisoniers.arrestation.index')->with('success', 'Arrestation enregistrée avec succès');

    }

    public function store(Request $request)
    {
        // Validation des données envoyées
        $validatedData = $request->validate([
            'nom' => 'required|string|max:255',
            'prenom' => 'required|string|max:255',
            'matricule' => 'required|string|max:191|unique:prisoniers,matricule',
            'sexe' => 'required|in:Homme,Femme',
            'date_naissance' => 'nullable|date',
            'lieu_naissance' => 'nullable|string|max:255',
            'nationalite' => 'nullable|string|max:255',
            'statut_matrimonial' => 'nullable|in:Célibataire,Marié(e),Divorcé(e),Veuf(ve)',
            'photo' => 'nullable|image|max:2048', // max 2 Mo
            'date_incarceration' => 'nullable|date',
            'date_liberation_prevue' => 'nullable|date|after_or_equal:date_incarceration',
            'motif' => 'nullable|string|max:255',
            'statut' => 'required|in:Arrêté,Libéré,Transféré',
        ]);

        // Gestion upload photo si présente
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('photos_prisonniers', 'public');
            $validatedData['photo'] = $path;
        }

        // Création du prisonnier en base
        Prisonier::create($validatedData);

        // Redirection avec message succès
        return redirect()->route('prisoniers.index')->with('success', 'Prisonnier enregistré avec succès.');
    }

    public function edit($id)
    {
        $prison = Prison::findOrFail($id);

        $quartiers = Quartier::all();

        // On récupère le type pour la prison actuelle
        $type = $prison->type;

        if($type=='pa' || $type=='PA') {
            $type="cp";
        }
        //dd($type);
        // On sélectionne les utilisateurs dont le rôle correspond au type
        $directeurs = User::where('role', $type)->get();

        return view('prisons.edit', compact('prison', 'quartiers', 'directeurs', 'type'));
    }

    public function detail($id)
    {
        $prisonier = Prisonier::findOrFail($id);

        $arrestationCount = $prisonier->arrestations()->count();
        $crimeCount = $prisonier->crimes()->count();

        $arrestationsParMois = $prisonier->arrestations()
            ->selectRaw("DATE_FORMAT(created_at, '%Y-%m') as mois, COUNT(*) as total")
            ->groupBy('mois')->orderBy('mois')->get();

        $roles = DB::table('crime_prisonnier')
            ->where('prisonnier_id', $prisonier->id)
            ->select('rôle', DB::raw('COUNT(*) as total'))
            ->groupBy('rôle')
            ->get();

        $crimes = $prisonier->crimes()->withPivot(['rôle', 'date_implication', 'condamnation'])->get();
        $arrestations = $prisonier->arrestations()->with('prison')->get();

        $dureeDetention = $prisonier->date_incarceration 
            ? now()->diffInDays($prisonier->date_incarceration) 
            : null;

        
        return view('prisoniers.detail', compact('prisonier', 'arrestationCount', 'crimeCount', 'arrestationsParMois', 'roles', 'crimes', 'arrestations', 'dureeDetention'));
    }

    public function generate($id)
    {
        $arrestation = Arrestation::with('prisonier')->findOrFail($id);
        $prisonnier = $arrestation->prisonier;

        $pdf = Pdf::loadView('prisoniers.proces_verbal', [
            'pv_number' => 'PV-' . Str::upper(Str::random(6)),
            'poste' => $arrestation->prison->name,
            'date' => now(),
            'heure' => now()->format('H:i'),
            'chef' => Auth::user()->name,
            'grade' => Auth::user()->role,
            'matricule' => 'POL-0123',
            'prisonnier' => $prisonnier,
            'arrestation' => $arrestation,
        ]);

        return $pdf->download('proces-verbal-arrestation.pdf');
        // ou ->stream() pour affichage direct
    }

    public function prisonier_arrestation_edit($id)
    {
        $arrestation = Arrestation::with('crimes')->findOrFail($id);
        $prisons = Prison::all();
        $users = User::all();
        $crime_prisoniers =CrimePrisonier::where('prisonnier_id', $arrestation->prisonier->id)->get();

        return view('prisoniers.arrestations.edit', compact('arrestation', 'prisons', 'users', 'crime_prisoniers'));
    }

    public function prisonier_arrestation_update(Request $request, $id)
    {
         $request->validate([
            'date_arrestation' => 'required|date',
            'prison_id' => 'required|exists:prisons,id',
            'prisonier_id' => 'required|exists:prisoniers,id',
            'responsable_id' => 'required|exists:users,id',
            'statut' => 'required|string',
            'date_relachement' => 'nullable|date',
            'crimes' => 'array',
            'crimes.*' => 'exists:crimes,id',
        ]);

        $arrestation = Arrestation::findOrFail($id);
        $arrestation->update([
            'date_arrestation' => $request->date_arrestation,
            'prison_id' => $request->prison_id,
            'prisonier_id' => $request->prisonier_id,
            'responsable_id' => $request->responsable_id,
            'statut' => $request->statut,
            'date_relachement' => $request->date_relachement,
        ]);
  
        // Mettre à jour les crimes associés
        $arrestation->crimes()->sync($request->crimes);

        return redirect()->route('prisoniers.arrestation.index')->with('success', 'Arrestation mise à jour avec succès');

    }
    public function update(Request $request, $id)
    {
        $prison = Prison::findOrFail($id);

        $validated = $request->validate([
            'code' => 'nullable|string|max:255',
            'name' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'user_id' => 'required|exists:users,id',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'date_ouverture' => 'nullable|date',
            'is_active' => 'required|numeric|in:0,1',
            'latitude' => 'nullable|string',
            'longitude' => 'nullable|string',
            'capacity' => 'required|integer',
            'location' => 'nullable|string|max:255',
            'quartier_id' => 'required|exists:quartiers,id',
        ]);

        $prison->update($validated);

        return redirect()->route('prisons.index')->with('success', 'Prison mise à jour avec succès.');
    }

    public function stat()
    {
        $totalPrisons = Prison::count();

        $prisonsByType = Prison::select('type', DB::raw('count(*) as total'))
            ->groupBy('type')
            ->pluck('total', 'type');

        $topQuartier = Prison::join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->select('quartiers.name', DB::raw('count(*) as total'))
            ->groupBy('quartiers.name')
            ->orderByDesc('total')
            ->first();

        $largestPrison = Prison::orderByDesc('capacity')->first();

        $prisonsByStatus = Prison::select('is_active', DB::raw('count(*) as total'))
            ->groupBy('is_active')
            ->pluck('total', 'is_active');

            $title="Les statistiques";

            //dd($prisonsByType);

        return view('prisons.stat', compact(
            'totalPrisons', 'prisonsByType', 'topQuartier', 'largestPrison', 'prisonsByStatus','title'
        ));
    }



}
