<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Quartier;
use App\Models\Prison;
use Illuminate\Support\Facades\DB;

class PrisonController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $prisons = Prison::with([
            'quartier.arrondissement.zone.ville'
        ])->get();

        $title="Listes des commissariats / Postes";

        return view('prisons.index', compact('prisons','title'));
    }
    public function cc(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'CC')->get();

        $title="Commisairiats centraux";
        
        return view('prisons.index', compact('prisons',"title"));
    }
    public function cpa(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'CPA')->get();

        $title="Commisairiats des polices d'arrondissement";
        
        return view('prisons.index', compact('prisons',"title"));
    }
    public function cpq(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'CPQ')->get();

        $title="Commisairiats des polices de quartier";
        
        return view('prisons.index', compact('prisons',"title"));
    }
    public function pa(Request $request)
    {
        $prisons = Prison::with(['quartier.arrondissement.zone.ville'])->where('type', 'PA')->get();

        $title="Les postes avancés";
        
        return view('prisons.index', compact('prisons',"title"));
    }

    // fonction create
    public function create($type)
    {
        
        $quartiers = Quartier::all();

        $role=$type;
        
        if($type=='pa') {
            $role="cp";
        }
        // Récupérer les utilisateurs qui ont pour rôle le type choisi
        $directeurs = \App\Models\User::where('role', $role)->get();
        
        return view('prisons.create', compact('quartiers','type','directeurs'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'code' => 'required|string|unique:prisons',
            'name' => 'required|string|max:255',
            'type' => 'required|string|in:cc,cpa,cpq,pa',
            'capacity' => 'required|integer|min:1',
            'location' => 'nullable|string|max:255',
            'quartier_id' => 'required|exists:quartiers,id',
            'user_id' => 'required|exists:users,id',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'is_active' => 'required|numeric|in:0,1',
            // ajoute d'autres validations si besoin
        ]);

        $prison = new \App\Models\Prison();
        $prison->name = $request->name;
        $prison->is_active = $request->is_active;
        $prison->type = $request->type;
        $prison->capacity = $request->capacity;
        $prison->quartier_id = $request->quartier_id;
        $prison->user_id = $request->user_id; // peut être null
        $prison->code = $request->code;
        $prison->location = $request->location;
        $prison->latitude = $request->latitude;
        $prison->longitude = $request->longitude;
        $phone->phone = $request->phone;
        $phone->email = $request->email;

        $prison->save();

        return redirect()->route('prisons.index')->with('success', 'Prison créée avec succès.');
    }

    public function edit($id)
    {
        $prison = Prison::findOrFail($id);

        $quartiers = Quartier::all();

        // On récupère le type pour la prison actuelle
        $type = $prison->type;

        if($type=='pa' || $type=='PA') {
            $type="cp";
        }
        //dd($type);
        // On sélectionne les utilisateurs dont le rôle correspond au type
        $directeurs = User::where('role', $type)->get();

        return view('prisons.edit', compact('prison', 'quartiers', 'directeurs', 'type'));
    }

    public function update(Request $request, $id)
    {
        $prison = Prison::findOrFail($id);

        $validated = $request->validate([
            'code' => 'nullable|string|max:255',
            'name' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'user_id' => 'required|exists:users,id',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'date_ouverture' => 'nullable|date',
            'is_active' => 'required|numeric|in:0,1',
            'latitude' => 'nullable|string',
            'longitude' => 'nullable|string',
            'capacity' => 'required|integer',
            'location' => 'nullable|string|max:255',
            'quartier_id' => 'required|exists:quartiers,id',
        ]);

        $prison->update($validated);

        return redirect()->route('prisons.index')->with('success', 'Prison mise à jour avec succès.');
    }

    public function detail($id)
    {
        $prison = Prison::findOrFail($id);
        // Widgets
        $arrestationCount = $prison->arrestations()->count();
        $materielCount = $prison->materielPrisons()->count();
        $interventionCount = $prison->operations()->count();
        $userCount = $prison->users()->count();

        // Top 5 arrestations récentes
        $recentArrestations = $prison->arrestations()->latest()->take(5)->get();

        // Derniers matériels
        $recentMateriels = $prison->materielPrisons()->latest()->take(5)->get();

        // Dernières interventions
        $recentInterventions = $prison->operations()->latest()->take(5)->get();

        // Graphique : arrestations par mois
        $arrestationsMonthly = $prison->arrestations()
            ->select(DB::raw("DATE_FORMAT(created_at, '%Y-%m') AS mois"), DB::raw('count(*) AS total'))
            ->groupBy('mois')->orderBy('mois')->get();

        // Chart matériel : count par type matériel
        $materielParType = $prison->materielPrisons()
            ->select('type', DB::raw('count(*) as total'))
            ->groupBy('type')->get();


        return view('prisons.detail', compact(
            'prison',
            'arrestationCount',
            'materielCount',
            'interventionCount',
            'userCount',
            'recentArrestations',
            'recentMateriels',
            'recentInterventions',
            'arrestationsMonthly',
            'materielParType'
        ));
    }



    public function stat()
    {
        $totalPrisons = Prison::count();

        $prisonsByType = Prison::select('type', DB::raw('count(*) as total'))
            ->groupBy('type')
            ->pluck('total', 'type');

        $topQuartier = Prison::join('quartiers', 'prisons.quartier_id', '=', 'quartiers.id')
            ->select('quartiers.name', DB::raw('count(*) as total'))
            ->groupBy('quartiers.name')
            ->orderByDesc('total')
            ->first();

        $largestPrison = Prison::orderByDesc('capacity')->first();

        $prisonsByStatus = Prison::select('is_active', DB::raw('count(*) as total'))
            ->groupBy('is_active')
            ->pluck('total', 'is_active');

            $title="Les statistiques";

            //dd($prisonsByType);

        return view('prisons.stat', compact(
            'totalPrisons', 'prisonsByType', 'topQuartier', 'largestPrison', 'prisonsByStatus','title'
        ));
    }



}
