<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Voiture;
use Illuminate\Support\Facades\DB;

class CarController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');

        $cars = Voiture::with('proprietaire')
        ->when($search, function ($query, $search) {
            return $query->where('marque', 'like', '%' . $search . '%')
                        ->orWhere('modele', 'like', '%' . $search . '%')
                        ->orWhere('immatriculation', 'like', '%' . $search . '%')
                        ->orWhere('carburant', 'like', '%' . $search . '%');
        })
        ->paginate(10);

        return view('cars.index', compact('cars'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $users = User::all(); // Pour la liste des propriétaires
        return view('cars.create', compact('users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {   
        // 🔍 Validation des champs
        $validated = $request->validate([
            'marque'           => 'required|string|max:255',
            'modele'           => 'required|string|max:255',
            'annee'            => 'required|integer|min:1900|max:' . date('Y'),
            'immatriculation'  => 'required|string|max:100|unique:cars,immatriculation',
            'carburant'        => 'required|string',
            'transmission'     => 'required|string',
            'kilometrage'      => 'nullable|integer|min:0',
            'nombre_places'    => 'nullable|integer|min:1|max:20',
            'description'      => 'nullable|string',
            'statut'           => 'required|string',
            'image'            => 'nullable|image|max:2048', // max 2MB
            'id_proprietaire'  => 'required|exists:users,id',
        ]);

        // 📷 Gérer le fichier image s'il existe
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('cars', 'public');
            $validated['image'] = $imagePath;
        }

        // 💾 Créer le véhicule
        Car::create($validated);

        // ✅ Redirection avec succès
        return redirect()->route('cars.index')->with('success', 'Véhicule ajouté avec succès.');

    }


    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }
    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $users = User::all();

        $chauffeurs = User::where('type', 2)->get(); // <- filtrage par type;

        $car = Voiture::findOrFail($id);
        return view('cars.edit', ['car' => $car, 'users' => $users, 'chauffeurs' => $chauffeurs]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // 🔍 Validation des champs
        $validated = $request->validate([
            'marque'           => 'required|string|max:255',
            'modele'           => 'required|string|max:255',
            'annee'            => 'required|integer|min:1900|max:' . date('Y'),
            'immatriculation'  => 'required|string|max:100|unique:voitures,immatriculation,' . $id,
            'carburant'        => 'required|string',
            'transmission'     => 'required|string',
            'kilometrage'      => 'nullable|integer|min:0',
            'nombre_places'    => 'nullable|integer|min:1|max:20',
            'description'      => 'nullable|string',
            'statut'           => 'required|string',
            'image'            => 'nullable|image|max:2048',
            'id_proprietaire'  => 'required|exists:users,id',
        ]);

        // 2. Récupération de l'utilisateur
        $car = Voiture::findOrFail($id);

        // 📷 Gestion de la nouvelle image (si une nouvelle image est uploadée)
        if ($request->hasFile('image')) {
            // Supprimer l'ancienne image si elle existe
            if ($car->image && Storage::disk('public')->exists($car->image)) {
                Storage::disk('public')->delete($car->image);
            }

            // Enregistrer la nouvelle image
            $validated['image'] = $request->file('image')->store('cars', 'public');
        }

        

        $car->id_user=$request->id_proprietaire;

        //dd($car->id_user);
        // 💾 Mise à jour des données
        $car->update($validated);

        // ✅ Redirection
        return redirect()->route('cars.index')->with('success', 'Véhicule mis à jour avec succès.');
    }


    public function affecterChauffeur(Request $request, $vehiculeId)
    {
        $request->validate([
            'chauffeur_id' => 'required|exists:users,id',
        ]);

        $car = Voiture::findOrFail($vehiculeId);
        $car->id_chauffeur = $request->input('chauffeur_id');
        $car->save();

        return redirect()->back()->with('success', 'Chauffeur affecté au véhicule avec succès.');
    }

    

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $car = Voiture::findOrFail($id);
        $car->delete();
        return redirect()->route('cars.index')->with('success', 'Utilisateur supprimé avec succès.');
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'integer|exists:voitures,id',
        ]);

        Voiture::whereIn('id', $request->ids)->delete();

        return redirect()->route('cars.index')->with('success', 'Vehicule supprimés avec succès.');
    }

    public function stat()
    {
        $totalCars = Voiture::count();
        $carsActifs = Voiture::where('statut', 1)->count();
        $carsInactifs = Voiture::where('statut', 0)->count();

        $carsParMois = Voiture::select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m") as mois'),
            DB::raw('COUNT(*) as total')
        )->groupBy('mois')->orderBy('mois')->get();

        $carsParGamme = Voiture::join('gammes_vehicules', 'voitures.gamme_id', '=', 'gammes_vehicules.id')
            ->select('gammes_vehicules.libelle as gamme', DB::raw('COUNT(*) as total'))
            ->groupBy('gammes_vehicules.libelle')->get();

        $carsParCarburant = Voiture::select('carburant', DB::raw('COUNT(*) as total'))
            ->groupBy('carburant')->get();

        $carsParStatut = Voiture::select(
            DB::raw('CASE WHEN statut = 1 THEN "Actif" ELSE "Inactif" END as statut'),
            DB::raw('COUNT(*) as total')
        )->groupBy('statut')->get();

        return view('cars.stat', compact(
            'totalCars',
            'carsActifs',
            'carsInactifs',
            'carsParMois',
            'carsParGamme',
            'carsParCarburant',
            'carsParStatut'
        ));
    }

}
